# encoding: utf-8
from __future__ import annotations

from typing import Optional

import click

import ckan.model as model
from ckan.cli import error_shout
from ckan.cli.user import add_user


@click.group(
    short_help=u"Gives sysadmin rights to a named user.",
    invoke_without_command=True,
)
@click.pass_context
def sysadmin(ctx: click.Context):
    """Gives sysadmin rights to a named user.

    """
    if ctx.invoked_subcommand is None:
        ctx.invoke(list_sysadmins)


@sysadmin.command(name=u"list", help=u"List sysadmins.")
def list_sysadmins():
    click.secho(u"Sysadmins:")
    sysadmins = model.Session.query(model.User).filter_by(
        sysadmin=True, state=u"active"
    )
    click.secho(u"count = %i" % sysadmins.count())
    for sysadmin in sysadmins:
        click.secho(
            u"%s name=%s email=%s id=%s"
            % (
                sysadmin.__class__.__name__,
                sysadmin.name,
                sysadmin.email,
                sysadmin.id,
            )
        )


@sysadmin.command(help=u"Convert user into a sysadmin.")
@click.argument("username")
@click.argument("args", nargs=-1)
@click.option("--create", is_flag=True, help="Create user if it doesn't exist")
@click.pass_context
def add(
        ctx: click.Context,
        username: str,
        args: list[str],
        create: Optional[bool] = False):
    user = model.User.by_name(str(username))
    if not user:
        if not create:
            click.secho(u'User "%s" not found' % username, fg=u"red")
            if click.confirm(
                u"Create new user: %s?" % username, default=True, abort=True
            ):
                create = True
        if create:
            ctx.params.pop("create", None)
            ctx.forward(add_user)
            user = model.User.by_name(str(username))
    assert user
    user.sysadmin = True
    model.Session.add(user)
    model.repo.commit_and_remove()
    click.secho(u"Added %s as sysadmin" % username, fg=u"green")


@sysadmin.command(help=u"Removes user from sysadmins.")
@click.argument(u"username")
def remove(username: str):
    user = model.User.by_name(str(username))
    if not user:
        return error_shout(u'Error: user "%s" not found!' % username)
    user.sysadmin = False
    model.repo.commit_and_remove()
    click.secho(u"Removed %s from sysadmins" % username, fg=u"green")
